<?php
/* --------------------------------------------------------------
   FlysystemAdapter.php 2020-04-29
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2019 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Core\Filesystem;

use Gambio\Core\Filesystem\Exceptions\FilesystemException;
use Gambio\Core\Filesystem\Interfaces\Filesystem;
use League\Flysystem\FileExistsException;
use League\Flysystem\FileNotFoundException as FlysystemFileNotFoundException;
use League\Flysystem\FilesystemInterface as Flysystem;

/**
 * Class Filesystem
 *
 * @package Gambio\Core\Filesystem
 */
class FlysystemAdapter implements Filesystem
{
    /**
     * @var Flysystem
     */
    protected $flysystem;
    
    /**
     * @var string
     */
    protected $baseDirectory;
    
    
    /**
     * FilesystemAdapter constructor.
     *
     * @param Flysystem $flysystem
     * @param string    $baseDirectory
     */
    public function __construct(Flysystem $flysystem, string $baseDirectory)
    {
        $this->flysystem     = $flysystem;
        $this->baseDirectory = $baseDirectory;
    }
    
    
    /**
     * @inheritDoc
     */
    public function getBaseDirectory(): string
    {
        return $this->baseDirectory;
    }
    
    
    /**
     * @inheritDoc
     */
    public function has(string $path): bool
    {
        return $this->flysystem->has($path);
    }
    
    
    /**
     * @inheritDoc
     */
    public function read(string $path): ?string
    {
        try {
            $content = $this->flysystem->read($path);
            if ($content === false) {
                return null;
            }
            
            return $content;
        } catch (FlysystemFileNotFoundException $exception) {
            return null;
        }
    }
    
    
    /**
     * @inheritDoc
     */
    public function list(string $directory, bool $recursive = false): \Gambio\Core\Filesystem\Interfaces\DirectoryItems
    {
        $files = [];
        foreach ($this->flysystem->listContents($directory, $recursive) as $fileData) {
            $files[] = DirectoryItem::createFromFlysystem($fileData,
                                                          $this->baseDirectory);
        }
        
        return DirectoryItems::create(...$files);
    }
    
    
    /**
     * @inheritDoc
     */
    public function create(string $path, string $content = ''): void
    {
        try {
            if ($this->flysystem->write($path, $content) === false) {
                throw new FilesystemException('Could not create file ' . $path);
            }
        } catch (FileExistsException $exception) {
            throw new FilesystemException('File already exists');
        }
    }
    
    
    /**
     * @inheritDoc
     */
    public function update(string $path, string $content): void
    {
        if ($this->flysystem->put($path, $content) === false) {
            throw new FilesystemException('Could not update file ' . $path);
        }
    }
    
    
    /**
     * @inheritDoc
     */
    public function delete(string $path): void
    {
        try {
            if ($this->flysystem->delete($path) === false) {
                throw new FilesystemException('Could not delete file ' . $path);
            }
        } catch (FlysystemFileNotFoundException $exception) {
        }
    }
}